/**
 * \file: exchnd_backend_errmem.c
 *
 * Implementation of a backend that dumps exception handler
 * messages into error memory.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <errno.h>
#include <limits.h>
#include <pthread.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "exchnd_backend.h"
#include "exchnd_interface.h"

/*
 * Implementation of an exception handler backend that dumps exception
 * messages to a file.
 */

typedef struct ExchndBackendErrmem {
    ExchndBackend_t base;     /* exception handler backend interface */
    FILE *fd;
} ExchndBackendErrmem_t;

/*
 * \func retrieve_msg
 *
 * Retrieve message from msg_list and store it in the backend.
 *
 * \param backend Pointer to the original backend.
 */
static void *retrieve_msg(void *backend)
{
    ExchndBackendErrmem_t *errmem_backend = (ExchndBackendErrmem_t *)backend;
    struct msg_list *current = mlist;
    char *msg = NULL;
    pthread_mutex_t *msg_lck = &mlist->msg_lck;

    pthread_mutex_lock(msg_lck);
    prctl(PR_SET_NAME, "be_errmem", NULL, NULL, NULL);

#if (DEFAULT_BACKEND & ERRMEM_BACKEND)
    /* Default backend ready to receive datas. */
    pthread_cond_broadcast(&msg_handled);
#endif

    pthread_cond_wait(&new_msg, msg_lck);

    while (!(action & EXCHND_TERMINATE)) {
        current = mlist->next;

        while (current != mlist) {
            if (current->read_flag & ERRMEM_BACKEND) {
                current = current->next;
                continue;
            }

            msg = current->msg;
            /* No need to keep the lock during write operation */
            pthread_mutex_unlock(msg_lck);
            errmem_backend->base.store(&errmem_backend->base,
                                       current->len, msg);

            /* Get lock again to update message structure */
            pthread_mutex_lock(msg_lck);
            current->read_flag |= ERRMEM_BACKEND;
            current = current->next;
        }

#if (DEFAULT_BACKEND & ERRMEM_BACKEND)
        /* List complete for default backend */
        pthread_cond_broadcast(&msg_handled);
#endif

        pthread_cond_wait(&new_msg, msg_lck);
    }

    pthread_mutex_unlock(msg_lck);

    return NULL;
}

/*
 * \func store
 *
 * Write a message to the error memory
 *
 * \param backend Pointer to backend structure (must be errmem!)
 * \param len Length of the data
 * \param msg Data to be written to the error memory
 *
 * \return 0 if successful, error code otherwise
 */
static unsigned int store(struct ExchndBackend *backend,
                          int len, char *msg)
{
    ExchndBackendErrmem_t *errmem_backend = (ExchndBackendErrmem_t *)backend;
    (void)len;

    if ((errmem_backend != NULL) && (errmem_backend->fd != NULL))
        fprintf(errmem_backend->fd, "%s", msg);
    else
        return EINVAL;

    return 0;
}

/*
 * \func destroy
 *
 * Destroy the given error memory  backend structure and its contents.
 *
 * \param backend Pointer to backend structure (must be errmem!)
 */
static void destroy(struct ExchndBackend *backend)
{
    ExchndBackendErrmem_t *errmem_backend = (ExchndBackendErrmem_t *)backend;

    if (errmem_backend->fd != NULL)
        fclose(errmem_backend->fd);

    /* Wait until any pending message is processed. */
    pthread_join(errmem_backend->base.BackendThread, NULL);

    free(errmem_backend);
}

/*
 * \func exchnd_backend_create_errmem
 *
 * Create the errmem backend. The functions tries to open the errmem device
 *
 * \param next Pointer to linked list of backends
 *
 * \return pointer to new backend or NULL memory allocation failed
 */
ExchndBackend_t *exchnd_backend_create_errmem(ExchndBackend_t *next)
{
    ExchndBackendErrmem_t *errmem_backend =
        calloc(1, sizeof(ExchndBackendErrmem_t));

    if (errmem_backend != NULL) {
        int flags;
        int fd = 0;
        /* setup the exception handler backend interface */
        errmem_backend->base.store = store;
        errmem_backend->base.destroy = destroy;
        errmem_backend->base.next = next;

        /* Use default error memory device */
        errmem_backend->fd = fopen("/dev/errmem", "w");

        if (errmem_backend->fd == NULL) {
            free(errmem_backend);
            exchnd_print_error("Could not open /dev/errmem. Error %d",
                               errno);
            return next;
        }

        /* No buffering, sync on write */
        fd = fileno(errmem_backend->fd);

        if (fd) {
            setvbuf(errmem_backend->fd, NULL, _IONBF, 0);
            flags = fcntl(fd, F_GETFL) | O_DSYNC;

            fcntl(fd, F_SETFL, flags);
        }

        if (exchnd_create_thread(&errmem_backend->base.BackendThread,
                                 retrieve_msg,
                                 errmem_backend,
                                 0))
            return next;

        /* We assume base to be the first element. */
        return (ExchndBackend_t *)errmem_backend;
    } else {
        return next;
    }
}